class BackgroundService {
  constructor() {
    this.init();
  }

  init() {
    this.setupInstallHandler();
    this.setupMessageHandlers();
  }

  setupInstallHandler() {
    chrome.runtime.onInstalled.addListener((details) => {
      if (details.reason === 'install') {
        this.handleFirstInstall();
      }
    });
  }

  async handleFirstInstall() {
    await chrome.storage.local.set({
      userData: {}
    });

    chrome.tabs.create({
      url: chrome.runtime.getURL('settings.html')
    });
  }

  setupMessageHandlers() {
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      this.handleMessage(request, sender, sendResponse);
      return true; // Keep message channel open for async response
    });
  }

  async handleMessage(request, sender, sendResponse) {
    try {
      switch (request.action) {
        case 'getData':
          const data = await this.getData();
          sendResponse({ success: true, data });
          break;

        case 'saveData':
          await this.saveData(request.data);
          sendResponse({ success: true });
          break;

        case 'openPopup':
          await this.openPopup();
          sendResponse({ success: true });
          break;

        default:
          sendResponse({ success: false, error: 'Unknown action' });
      }
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
  }

  async openPopup() {
    const popupUrl = chrome.runtime.getURL('popup.html');
    const window = await chrome.windows.create({
      url: popupUrl,
      type: 'popup',
      width: 400,
      height: 600
    });
  }

  async getData() {
    try {
      const result = await chrome.storage.local.get(['userData']);
      return {
        userData: result.userData || {}
      };
    } catch (error) {
      return {
        userData: {}
      };
    }
  }

  async saveData(data) {
    try {
      await chrome.storage.local.set(data);
    } catch (error) {
      throw new Error('Failed to save data');
    }
  }
}

new BackgroundService();
