class AutoFillEngine {
  constructor() {
    this.userData = {};
    this.hasAutoPopped = false;
    this.init();
  }

  async init() {
    try {
      await this.loadUserData();
      this.setupEventListeners();
      // Check for multiple inputs after 800ms delay
      setTimeout(() => {
        this.checkForMultipleInputs();
      }, 800);
    } catch (error) {
    }
  }

  async loadUserData() {
    try {
      const result = await chrome.storage.local.get(["userData"]);
      this.userData = result.userData || {};
    } catch (error) {
    }
  }

  setupEventListeners() {
    // Listen for DOM changes to detect dynamically added inputs
    const observer = new MutationObserver(() => {
      this.throttledCheckForMultipleInputs();
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true,
    });

    // Listen for close messages from the iframe
    window.addEventListener("message", (event) => {
      if (event.data && event.data.action === "closePopup") {
        const iframe = document.getElementById("autofill-popup-iframe");
        if (iframe) {
          iframe.remove();
        }
      }
    });

    // Listen for messages from the popup
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.action === "createHoveringPopup") {
        this.createHoveringPopup();
      }
    });
  }

  throttledCheckForMultipleInputs() {
    // Only check if we haven't already auto-popped for this page
    if (this.hasAutoPopped) return;

    // Add a small delay to avoid excessive checking
    setTimeout(() => {
      this.checkForMultipleInputs();
    }, 500);
  }

  checkForMultipleInputs() {
    try {
      // Only check if we haven't already auto-popped for this page
      if (this.hasAutoPopped) return;

      // Check if we're on a valid page
      if (!document.body || document.readyState !== "complete") {
        return;
      }

      const inputSelectors = [
        'input[type="text"]',
        'input[type="email"]',
        'input[type="tel"]',
        'input[type="password"]',
        'input[type="search"]',
        'input[type="url"]',
        "textarea",
      ];

      const allInputs = document.querySelectorAll(inputSelectors.join(", "));

      // Filter out hidden inputs and inputs that are not visible
      const visibleInputs = Array.from(allInputs).filter((input) => {
        try {
          const style = window.getComputedStyle(input);
          return (
            style.display !== "none" &&
            style.visibility !== "hidden" &&
            input.offsetWidth > 0 &&
            input.offsetHeight > 0 &&
            !input.disabled &&
            !input.readOnly
          );
        } catch (error) {
          return false;
        }
      });

      // If we find 2 or more visible input fields, trigger auto-popup
      if (visibleInputs.length >= 2) {
        this.hasAutoPopped = true;
        this.triggerAutoPopup();
      }
    } catch (error) {
    }
  }

  triggerAutoPopup() {
    try {
      // Show a visual indicator that the extension is available
      this.showExtensionIndicator();
    } catch (error) {
    }
  }

  showExtensionIndicator() {
    try {
      // Create a subtle indicator that the extension is available
      const indicator = document.createElement("div");
      indicator.id = "autofill-indicator";
      indicator.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: rgba(248, 209, 209, 0.9);
        backdrop-filter: blur(10px);
        -webkit-backdrop-filter: blur(10px);
        color: #333;
        padding: 10px 14px;
        border-radius: 8px;
        font-size: 16px;
        font-weight: 300;
        z-index: 10000;
        cursor: pointer;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.25);
        transition: all 0.3s ease;
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;
      `;
      indicator.textContent = "Autofill Ready";

      indicator.addEventListener("click", () => {
        this.createHoveringPopup();
        if (indicator.parentNode) {
          indicator.remove();
        }
      });

      document.body.appendChild(indicator);

      // Remove the indicator after 1500ms
      setTimeout(() => {
        if (indicator.parentNode) {
          indicator.remove();
        }
      }, 1500);
    } catch (error) {
    }
  }

  createHoveringPopup() {
    // Remove any existing popup
    const existingPopup = document.getElementById("autofill-popup-iframe");
    if (existingPopup) {
      existingPopup.remove();
    }

    const iframe = document.createElement("iframe");
    iframe.id = "autofill-popup-iframe";
    iframe.src = chrome.runtime.getURL("popup.html?isHovering=true");
    iframe.allow = "clipboard-write"; // Allow clipboard access

    // Updated styling to support dragging and resizing
    iframe.style.cssText = `
      position: fixed;
      top: 20%;
      left: 20%;
      width: 400px;
      height: 600px;
      min-width: 300px;
      min-height: 400px;
      border: none;
      border-radius: 12px;
      box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
      z-index: 2147483647;
      resize: both;
      overflow: auto;
    `;

    document.body.appendChild(iframe);

    // Drag functionality
    let isDragging = false;
    let offsetX = 0;
    let offsetY = 0;
    let lastX = 0;
    let lastY = 0;
    let ticking = false;

    // Listen for messages from the iframe to start dragging
    window.addEventListener("message", (event) => {
      if (event.source !== iframe.contentWindow) return;

      if (event.data.action === "startDrag") {
        isDragging = true;
        offsetX = event.data.x;
        offsetY = event.data.y;
        iframe.style.pointerEvents = "none"; // Disable mouse events on iframe
        document.body.style.userSelect = "none";
      }
    });

    // Handle dragging
    document.addEventListener("mousemove", (e) => {
      lastX = e.clientX;
      lastY = e.clientY;

      if (isDragging && !ticking) {
        window.requestAnimationFrame(() => {
          iframe.style.left = `${lastX - offsetX}px`;
          iframe.style.top = `${lastY - offsetY}px`;
          ticking = false;
        });
        ticking = true;
      }
    });

    // Stop dragging
    document.addEventListener("mouseup", () => {
      if (isDragging) {
        isDragging = false;
        iframe.style.pointerEvents = "auto"; // Re-enable mouse events on iframe
        document.body.style.userSelect = "";
      }
    });
  }
}

// Initialize the engine
new AutoFillEngine();
