class PopupController {
  constructor() {
    this.userData = {};
    this.isHovering = new URLSearchParams(window.location.search).get('isHovering') === 'true';
    this.init();
  }

  async init() {
    try {
      await this.loadData();
      this.setupEventListeners();
      this.renderData();
      // Hide empty state initially
      document.getElementById('empty-state').style.display = 'none';

      if (this.isHovering) {
        this.setupHoveringPopup();
      }
    } catch (error) {
    }
  }

  async loadData() {
    try {
      const result = await chrome.storage.local.get(['userData', 'customFields']);
      this.userData = result.userData || {};
      const customFields = result.customFields || [];
      customFields.forEach(field => {
        this.userData[field.name] = field.value;
      });
    } catch (error) {
    }
  }

  setupEventListeners() {
    document.getElementById('search-bar').addEventListener('input', (e) => {
      this.renderData(e.target.value);
    });

    document.getElementById('settings-btn').addEventListener('click', (e) => {
        e.preventDefault();
        chrome.tabs.create({ 
            url: chrome.runtime.getURL('settings.html') 
        });
    });

    document.getElementById('open-hovering-popup-btn').addEventListener('click', () => {
      chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
        chrome.tabs.sendMessage(tabs[0].id, {action: "createHoveringPopup"});
      });
    });
  }

  setupHoveringPopup() {
    const closeBtn = document.getElementById('close-btn');
    const header = document.querySelector('.header');

    closeBtn.style.display = 'block';
    closeBtn.addEventListener('click', () => {
      // Post a message to the parent window (the content script)
      window.parent.postMessage({ action: 'closePopup' }, '*');
    });

    header.addEventListener('mousedown', (e) => {
      // Prevent dragging from input fields
      if (e.target.tagName === 'INPUT') return;
      
      e.preventDefault();
      window.parent.postMessage({ 
        action: 'startDrag', 
        x: e.clientX, 
        y: e.clientY 
      }, '*');
    });

    
  }

  renderData(filter = '') {
    const dataList = document.getElementById('data-list');
    const emptyState = document.getElementById('empty-state');
    dataList.innerHTML = '';

    const searchTerm = filter.toLowerCase();
    const filteredEntries = Object.entries(this.userData).filter(([key, value]) => {
      if (!searchTerm) return true;
      return key.toLowerCase().includes(searchTerm) || 
             (String(value)).toLowerCase().includes(searchTerm);
    });

    if (filteredEntries.length === 0) {
      emptyState.style.display = 'block';
      dataList.style.display = 'none';
      if (filter) {
        emptyState.innerHTML = `
          <h3>No matches found</h3>
          <p>Try a different search term.</p>
        `;
      } else {
        emptyState.innerHTML = `
          <h3>No Data Found</h3>
          <p>Add data in the settings to get started.</p>
        `;
      }
      return;
    }

    emptyState.style.display = 'none';
    dataList.style.display = 'flex';

    filteredEntries.forEach(([key, value]) => {
      const item = document.createElement('div');
      item.className = 'data-item';
      
      const highlightedKey = this.highlightText(key, filter);
      const highlightedValue = this.highlightText(String(value), filter);

      item.innerHTML = `
        <div class="data-item-info">
          <div class="data-item-key">${highlightedKey}</div>
          <div class="data-item-value">${highlightedValue}</div>
        </div>
        <button class="copy-btn" data-value="${String(value)}">Copy</button>
      `;
      
      dataList.appendChild(item);
    });

    dataList.querySelectorAll('.copy-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const valueToCopy = e.currentTarget.getAttribute('data-value');
        this.copyToClipboard(valueToCopy, e.currentTarget);
      });
    });
  }

  copyToClipboard(text, button) {
    const textArea = document.createElement("textarea");
    textArea.value = text;
    document.body.appendChild(textArea);
    textArea.select();
    try {
      document.execCommand("copy");
      const originalText = button.textContent;
      button.textContent = "Copied!";
      button.classList.add("copied");
      setTimeout(() => {
        button.textContent = originalText;
        button.classList.remove("copied");
      }, 2200);
    } catch (err) {
    }
    document.body.removeChild(textArea);
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  highlightText(text, searchTerm) {
    if (!searchTerm || !text) {
      return this.escapeHtml(text);
    }
    
    const escapedText = this.escapeHtml(text);
    const escapedSearchTerm = this.escapeHtml(searchTerm);
    const regex = new RegExp(`(${escapedSearchTerm})`, 'gi');
    
    return escapedText.replace(regex, '<mark style="background-color: #ef4444; color: white; padding: 1px 2px; border-radius: 2px;">$1</mark>');
  }
}

document.addEventListener('DOMContentLoaded', () => {
  new PopupController();
});

// Loading animation with error handling
document.addEventListener('DOMContentLoaded', function() {
    const loadingOverlay = document.getElementById('loadingOverlay');
    const mainContent = document.getElementById('mainContent');
    
    // Check if elements exist before manipulating them
    if (loadingOverlay && mainContent) {
        setTimeout(() => {
            loadingOverlay.classList.add('hidden');
            mainContent.classList.add('visible');
            
            // Remove loading overlay after transition
            setTimeout(() => {
                if (loadingOverlay) {
                    loadingOverlay.style.display = 'none';
                }
            }, 500);
        }, 100);
    } else {
        // Fallback: show main content immediately if elements are missing
        if (mainContent) {
            mainContent.classList.add('visible');
        }
        if (loadingOverlay) {
            loadingOverlay.style.display = 'none';
        }
    }
});