class SettingsController {
  constructor() {
    this.userData = {};
    this.customFields = [];
    this.formFields = [
      'firstName', 'lastName', 'fullName', 'email', 'phone', 'mobile',
      'website', 'linkedin', 'address', 'city', 'state', 'postalCode',
      'country', 'university', 'college', 'degree', 'major', 'graduationYear',
      'gpa', 'company', 'jobTitle', 'experience', 'skills'
    ];
    this.presets = {
      basic: {
        firstName: 'First Name',
        lastName: 'Last Name',
        fullName: 'Full Name',
        email: 'Email Address'
      },
      contact: {
        phone: 'Phone Number',
        mobile: 'Mobile Number',
        website: 'Website',
        linkedin: 'LinkedIn Profile'
      },
      address: {
        address: 'Street Address',
        city: 'City',
        state: 'State/Province',
        postalCode: 'Postal Code',
        country: 'Country'
      },
      education: {
        university: 'University',
        college: 'College',
        degree: 'Degree',
        major: 'Major',
        graduationYear: 'Graduation Year',
        gpa: 'GPA'
      }
    };
    this.init();
  }

  async init() {
    await this.loadState();
    this.setupEventListeners();
    this.updateUI();
  }

  async loadState() {
    try {
      const result = await chrome.storage.local.get(['userData', 'customFields']);
      this.userData = result.userData || {};
      this.customFields = result.customFields || [];
    } catch (error) {
      console.error("Error loading state:", error);
    }
  }
  
  setupEventListeners() {
    // Form management
    document.getElementById('save-btn').addEventListener('click', () => {
      this.saveFormData();
    });

    document.getElementById('add-custom-field-btn').addEventListener('click', () => {
      this.addCustomField();
    });

    // Preset buttons
    document.querySelectorAll('.preset-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        this.addPreset(e.currentTarget.dataset.preset);
      });
    });

    // Auto-fill full name when first and last name are entered
    document.getElementById('firstName').addEventListener('input', () => {
      this.updateFullName();
    });

    document.getElementById('lastName').addEventListener('input', () => {
      this.updateFullName();
    });
  }

  updateFullName() {
    const firstName = document.getElementById('firstName').value.trim();
    const lastName = document.getElementById('lastName').value.trim();
    const fullName = document.getElementById('fullName');
    
    if (firstName && lastName) {
      fullName.value = `${firstName} ${lastName}`;
    } else if (firstName || lastName) {
      fullName.value = firstName || lastName;
    }
  }

  addPreset(presetName) {
    const preset = this.presets[presetName];
    if (!preset) return;

    let addedCount = 0;
    Object.keys(preset).forEach(fieldId => {
      const field = document.getElementById(fieldId);
      if (field && !field.value.trim()) {
        // Don't overwrite existing values, just focus on empty fields
        field.focus();
        addedCount++;
      }
    });
    
    if (addedCount > 0) {
      this.showMessage(`Focusing on ${presetName} fields. Fill them out and click Save.`, 'success');
    } else {
      this.showMessage('All fields from this preset already have values.', 'info');
    }
  }

  addCustomField() {
    const nameInput = document.getElementById('customFieldName');
    const valueInput = document.getElementById('customFieldValue');
    const name = nameInput.value.trim();
    const value = valueInput.value.trim();

    if (!name || !value) {
      this.showMessage('Both field name and value are required for custom fields.', 'error');
      return;
    }

    // Prevent duplicate field names
    if (this.customFields.some(field => field.name.toLowerCase() === name.toLowerCase()) || this.formFields.includes(name)) {
      this.showMessage(`A field with the name "${name}" already exists.`, 'error');
      return;
    }

    this.customFields.push({ name, value });
    this.saveCustomFields();
    nameInput.value = '';
    valueInput.value = '';
    this.showMessage(`Custom field "${name}" added successfully.`, 'success');
  }

  async saveCustomFields() {
    await chrome.storage.local.set({ customFields: this.customFields });
    this.updateUI();
  }

  deleteCustomField(index) {
    if (index < 0 || index >= this.customFields.length) return;
    const fieldName = this.customFields[index].name;
    this.customFields.splice(index, 1);
    
    // Also remove from userData if it exists there
    if (this.userData[fieldName]) {
        delete this.userData[fieldName];
        chrome.storage.local.set({ userData: this.userData });
    }
    
    this.saveCustomFields();
    this.showMessage(`Custom field "${fieldName}" deleted.`, 'success');
  }

  async saveFormData() {
    const formData = {};
    let hasData = false;

    // Collect all form data
    this.formFields.forEach(fieldId => {
      const field = document.getElementById(fieldId);
      if (field && field.value.trim()) {
        formData[fieldId] = field.value.trim();
        hasData = true;
      }
    });

    // Add custom fields to formData
    this.customFields.forEach(field => {
        formData[field.name] = field.value;
        hasData = true;
    });

    if (!hasData) {
      this.showMessage('Please fill in at least one field before saving', 'error');
      return;
    }

    // Merge with existing data
    this.userData = { ...this.userData, ...formData };
    await chrome.storage.local.set({ userData: this.userData });

    this.showMessage(`Successfully saved ${Object.keys(formData).length} fields!`, 'success');
    this.updateUI();
  }

  renderData() {
    const dataList = document.getElementById('data-list');
    dataList.innerHTML = '';

    const allData = { ...this.userData };

    if (Object.keys(allData).length === 0 && this.customFields.length === 0) {
      dataList.innerHTML = '<div class="data-item"><div class="data-item-key">No Data</div><div class="data-item-value">No data saved yet. Fill out the form above and click Save.</div></div>';
      return;
    }

    Object.entries(allData).forEach(([key, value]) => {
      const item = document.createElement('div');
      item.className = 'data-item fade-in';
      item.innerHTML = `
        <div class="data-item-key">${this.escapeHtml(this.formatFieldName(key))}</div>
        <div class="data-item-value">${this.escapeHtml(String(value))}</div>
      `;
      dataList.appendChild(item);
    });
  }

  renderCustomFields() {
    const container = document.getElementById('custom-fields-container');
    if (!container) return;
    container.innerHTML = '';
  
    if (this.customFields.length === 0) {
      container.innerHTML = '<div class="data-item"><div class="data-item-key">No custom fields</div><div class="data-item-value">Add some using the form above.</div></div>';
      return;
    }
  
    this.customFields.forEach((field, index) => {
      const item = document.createElement('div');
      item.className = 'data-item fade-in';
      item.innerHTML = `
        <div class="data-item-key">${this.escapeHtml(field.name)}</div>
        <div class="data-item-value">${this.escapeHtml(field.value)}</div>
        <button class="btn btn-danger btn-small delete-custom-field-btn" data-index="${index}" style="position: absolute; right: 10px; top: 10px; padding: 5px 10px; font-size: 12px;">
            <svg width="12" height="12" fill="currentColor" viewBox="0 0 24 24">
                <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
            </svg>
        </button>
      `;
      container.appendChild(item);
    });
  
    document.querySelectorAll('.delete-custom-field-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const index = parseInt(e.currentTarget.dataset.index, 10);
        this.deleteCustomField(index);
      });
    });
  }

  formatFieldName(key) {
    // Convert camelCase to readable format
    return key
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, str => str.toUpperCase())
      .trim();
  }

  updateUI() {
    const dataDisplay = document.getElementById('data-display');
    const statusMessage = document.getElementById('status-message');

    // Hide status message initially
    statusMessage.style.display = 'none';

    dataDisplay.style.display = 'block';
    this.renderData();
    
    // Pre-fill the form with saved data
    this.formFields.forEach(fieldId => {
      const field = document.getElementById(fieldId);
      if (field && this.userData[fieldId]) {
        field.value = this.userData[fieldId];
      }
    });

    this.renderCustomFields();
  }

  showMessage(text, type = 'info') {
    const messageEl = document.getElementById('status-message');
    messageEl.textContent = text;
    messageEl.className = `status-message status-${type}`;
    messageEl.style.display = 'block';

    setTimeout(() => {
      messageEl.style.display = 'none';
    }, 3000);
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
}

document.addEventListener('DOMContentLoaded', () => {
  new SettingsController();
});
